{******************************************************************************
 *  init_mm.pp - initializes nucleOS physical memory manager
 *
 *
 *  Copyright (c) 2004 nucleOS Group [http://nucle-os.sourceforge.net/]
 *                                   [http://www.sf.net/projects/nucle-os]
 *                                   [http://www.saint-soft.de/nucleos/board/]
 *
 *  version 0.1 - 02/03/2004 - initial version
 *
 *  written by
 *    Michael Gerh"auser (saberrider) [saberrider@users.sourceforge.net]
 *
 *  This file does
 *   - initialize IDT
 *   - reinitialize GDT
 *   - reinitialize paging
 *   - nucleOS memmap L1 & L2
 *
 *  To be implemented
 *   - %
 *
 * This program is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software Foundation
 * (version 2, June 1991)
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program; if not, write to:
 * Free Software Foundation, Inc.
 * 59 Temple Place, Suite 330
 * Boston, MA 02111-1307 USA
 *
 ******************************************************************************}

//{$DEFINE DEBUG}
//{$DEFINE SMAP}

unit init_mm;


interface

uses convert, bits, strings, idt, mm, gfxdrvunit;

{* External procedures and functions *}
function setup_gdt: Pointer; external;
//procedure writeln( value: Pointer ); external;


{* External variables *}
var
   bGFXDRVCurrentX: byte; cvar; external;		{ this are the coordinates of the current text-output }
   bGFXDRVCurrentY: byte; cvar; external;


{* Exported variables *}
   Compiler:          String; cvar;
//   CompileTime:       String; cvar;
   GDT:               PLongInt; cvar;     // Pointer to GDT
   IDTBASE:           PLongInt; cvar; external;     // Pointer to IDT. defined in load_idt.s
   PAGE_DIRECTORY:    PLongInt; cvar;     // Pointer to Kernel Page Directory
   PT_KERNEL_DATA:    PLongInt; cvar;     // Pointer to Kernel .data Page Table
   PT_KERNEL_TEXT:    PLongInt; cvar;     // Pointer to Kernel .text Page Table
   PT_KERNEL_STACK:   PLongInt; cvar;     // Pointer to Kernel Stack Page Table
   PT_KERNEL_HEAP:    PLongInt; cvar;     // Pointer to Kernel .bss Page Table
   TOTAL_RAM:         LongInt; cvar;     // Amount of physical RAM installed
                                         // in kBytes.
   NO_MME_L1:         LongInt; cvar;     // Number of Memory Map Entries L1 available.
                                         // Calculated using amount of physical RAM
                                         // Maximum: 31768 (4GB)
   NO_MME_L2:         LongInt; cvar;     // Number of Memory Map Entries L2 available.
                                         // Maximum: 1024 (4GB)
   MEMORY_MAP_L1:     PPointer; cvar;
   MEMORY_MAP_L2:     PPointer; cvar;


{* Procedures and functions defined in this file *}

procedure mm_init;
procedure init_memmap;

{$ifdef SMAP}
procedure read_system_map;
{$endif}

procedure halt;


implementation


{* Types only used in this file *}
{$ifdef SMAP}
type
    PSMAP_Entry = ^NSMAP_Entry;
    NSMAP_Entry = record
         base_low:   DWord;
         base_hi:    DWord;
         len_low:    DWord;
         len_hi:     DWord;
         entry_type: DWord;
    end;
{$endif}

procedure halt; assembler; [public, alias:'HALT'];
asm
   cli
   hlt
end;


{$ifdef SMAP}
{******************************************************************************
 *   read_system_map
 ******************************************************************************
 *  does:
 *   reads system map, created by setup.s using int 15h
 ******************************************************************************}
procedure read_system_map;
var
    p:               PLongInt;
    smap_entries, i: LongInt;
    smap_entry:      PSMAP_Entry;
    tmp:             String;
begin
     p := $DC79C004;
     smap_entries := p^;

     {$IFDEF DEBUG}
     writeln( '' );
     writeln('Reading System Memory Map...');
     writeln('Number of Entries: ' + IntToStr( smap_entries ));

     {$ENDIF DEBUG}

     SMAP_Entry := $DC79C008;

     for i := 0 to smap_entries-1 do
     begin
          {$ifdef DEBUG}

          if SMAP_Entry^.entry_type = 1 then
             tmp := ' [available for OS]'
          else if SMAP_Entry^.entry_type = 2 then
             tmp := ' [reserved by System]'
          else
             tmp := ' [reserved]';

          writeln( IntToHexAligned( SMAP_Entry^.base_low, 8 ) + '-' + IntToHexAligned( SMAP_Entry^.base_low + SMAP_Entry^.len_low - 1, 8) + ' | Length: ' + IntToStr((SMAP_Entry^.len_low + 1) div 1024) + ' kB - Type: ' + IntToStr(SMAP_Entry^.entry_type) + tmp );

          {$endif DEBUG}
          writeln( '' );
          SMAP_Entry += 1;
      end;
end;
{$endif SMAP}

{******************************************************************************
 *   init_memmap;
 ******************************************************************************
 *  does:
 *   initializes our memory maps MEMORY_MAP_L1 and MEMORY_MAP_L2
 ******************************************************************************}
procedure init_memmap;
var
   i: LongInt;
begin
     NO_MME_L1 := TOTAL_RAM div 128;
     NO_MME_L2 := NO_MME_L1 div 32;

     MEMORY_MAP_L1 := $FFF6C000;
     MEMORY_MAP_L2 := $FFF8C000;

//   first 3 MB reserved:
     for i := 0 to 23 do
         	MEMORY_MAP_L1[i] := $FFFFFFFF;

//   3MB -> end of RAM is indicated free:
     for i := 24 to NO_MME_L1-1 do
         	MEMORY_MAP_L1[i] := 0;

//   first 3 MB reserved:
     MEMORY_MAP_L2[0] := $FFFFFF00;

//   3MB -> end of RAM is indicated free:
     for i := 1 to NO_MME_L2-1 do
         	MEMORY_MAP_L2[i] := 0;

//   has RAM in MB divided by 4 a remainder?
//   e.g.: 17MB RAM installed -> 17 = 1 mod 4
//   without fix below we wouldn't be able to map that MB
     if NO_MME_L1 mod 32 <> 0 then
     begin
         	// RAM in MB divided by 4 has a remainder!
         	// so we gotta fix memory_map_l2...
         	MEMORY_MAP_L2[NO_MME_L2] := 0;

         	for i := (31 - (NO_MME_L1 mod 32)) downto 0 do
         	    MEMORY_MAP_L2[NO_MME_L2] := set_bit(i, MEMORY_MAP_L2[NO_MME_L2]);

         	NO_MME_L2 += 1;

         	// ...and memory_map_l1
         	for i := NO_MME_L1 to NO_MME_L1+(32-(NO_MME_L1 mod 32))-1 do
         	    MEMORY_MAP_L1[i] := $FFFFFFFF;

         	NO_MME_L1 += 32-(NO_MME_L1 mod 32);
     end;
end;


{******************************************************************************
 *   mm_init
 ******************************************************************************
 *  does:
 *   initializes physical memory manager and page fault ISR
 ******************************************************************************}
procedure mm_init; [public, alias:'MM_INIT'];
var
    tmp:   String;
    p:     PLongInt;
    pb:    PByte;
begin
     init_idt;
     GDT := setup_gdt;

     writeln('Setting up Memory Manager');
     writeln( '' );

     IDTBASE := $FFF00500;
     {$IFDEF DEBUG}
     writeln( 'Reinitialized IDT...' );
     writeln( 'New IDT-Base: ' + IntToHexAligned ( LongInt( IDTBASE ), 8 ) );
     writeln( '' );
     {$ENDIF}

     {$IFDEF DEBUG}
     writeln( 'Reinitialized GDT...' );
     {$ENDIF}

     {$IFDEF DEBUG}
     writeln( 'New GDT-Location: ' + IntToHexAligned ( LongInt( GDT ), 8 ) );

     writeln( '' );
     {$ENDIF}

     p := $DC79C000;
     TOTAL_RAM := p^;
     {$IFDEF DEBUG}
     writeln('Total Ram: ' + IntToStr( TOTAL_RAM ) + ' kB');
     writeln('           ' + IntToStr( TOTAL_RAM div 1024 ) + ' MB');
     tmp := '"cleaned": ' + IntToStr( (TOTAL_RAM div 1024) * 1024 ) + ' kB';
     {$ELSE}
     tmp := 'Total Ram: ' + IntToStr( (TOTAL_RAM div 1024) ) + ' MB';
     {$ENDIF}

     {$ifdef DEBUG}
     TOTAL_RAM := (TOTAL_RAM div 1024) * 1024;
     {$endif}

     PT_KERNEL_TEXT  := $FFF04000;
     PT_KERNEL_DATA  := $FFF05000;
     PT_KERNEL_HEAP  := $FFF06000;
     PT_KERNEL_STACK := $FFF03000;
     PAGE_DIRECTORY  := $FFF07000;

     pb := @tmp;
     pb[0] := 40;
     mem_copy( $E0000000, @tmp+1, 40);
     Compiler := tmp;

     {$IFDEF SMAP}
     read_system_map();
     {$ENDIF SMAP}

     init_memmap();

     set_interrupt(14, @page_fault_isr);
end;

begin
end.
