{******************************************************************************
 *  mm.pp - nucleOS physical memory manager
 *
 *
 *  Copyright (c) 2004 nucleOS Group [http://nucle-os.sourceforge.net/]
 *                                   [http://www.sf.net/projects/nucle-os]
 *                                   [http://www.saint-soft.de/nucleos/board/]
 *
 *  version 0.1 - 02/03/2004 - first _very_ basic version
 *
 *  written by
 *    Michael Gerh"auser (saberrider) [saberrider@users.sourceforge.net]
 *
 *  This file does
 *   - page fault handling
 *   - get_free_page
 *   - get_free_dma_page
 *   - free_page
 *   - mem_copy
 *
 *  To be implemented
 *   - %
 *
 * This program is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software Foundation
 * (version 2, June 1991)
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program; if not, write to:
 * Free Software Foundation, Inc.
 * 59 Temple Place, Suite 330
 * Boston, MA 02111-1307 USA
 *
 ******************************************************************************}

//{$DEFINE DEBUG}

unit mm;

interface

uses convert, bits, gfxdrvunit;

{* External Procedures and functions *}
//procedure writeln( value: Pointer ); external;

{* Procedures and functions defined in this file *}
procedure mem_copy( src, dest: Pointer; length: LongInt );
procedure page_fault_ISR;
function get_pde_index( address: longint ): LongInt;
function get_pte_index( address: longint ): LongInt;
function get_free_page: Pointer;
function get_free_dma_page: Pointer;
procedure free_page( page: Pointer );
function mem_alloc( n: word ): Pointer;
function sbrk( n: word ): Pointer;
Procedure free( var p: Pointer );


type
    Pmem_header = ^mem_header;
    mem_header = record
      next: ^mem_header;
      size: Word;
    end;

var
   brk:                 LongInt; cvar;

   MEMORY_MAP_L1:       PPointer; cvar; external;
   MEMORY_MAP_L2:       PPointer; cvar; external;
   NO_MME_L1:           LongInt; cvar; external;
   NO_MME_L2:           LongInt; cvar; external;
   TOTAL_RAM:           LongInt; cvar; external;
   freelist:            mem_header; cvar; external;
   alloclist:           mem_header; cvar; external;

implementation

{******************************************************************************
 *   nucleOS page fault ISR
 ******************************************************************************
 *
 *  short description/explanation where to find needed informations:
 *
 *    faulting address is saved in cr2
 *
 *    stackframe after exception (see http://my.execpc.com/CE/AC/geezer/index.htm for more details):
 *
 *      ebp:           kernel's esp-backup
 *      ebp+4:         error code
 *      ebp+8:         EIP which caused exception
 *      ebp+12:        cs of faulting code
 *      ebp+16:        EFLAGS backup
 *
 *
 *  error-code:
 *
 *      see
 *            IA-32 Intel Architecture Software Developers Manual
 *                  Volume 3:  System Programming Guide
 *                            (page 184)
 *               for more information about page fault error-code
 *
 *      bit0: 0: not present
 *            1: protection violation
 *      bit1: 0: read fault
 *            1: write fault
 *      bit2: 0: cpr0-2
 *            1: cpr3
 *      bit3: 0: fault not caused by reserved bit violation
 *            1: fault caused by reserved bits set to 1 in page directory
 *
 *  to solve page fault
 *
 *      - get_free_page
 *      - update taks's cr3
 *      - remove error code ( add esp, 4 )
 *      - iret
 *
 ******************************************************************************}

procedure page_fault_ISR; [public, alias: 'PF_ISR'];
var
   tmp: String;
   err_addr, err_eip,
   err_code, pte_idx, pde_idx: LongInt;
   new_page: Pointer;
begin
     asm
        pushad              // push EAX, ECX, EDX, EBX, ESP, EBP, ESI and EDI
        sti                 // enable interrupts
        mov eax, [ebp+8]    // -+
        mov err_eip, eax    //  |
        mov eax, [ebp+4]    //  +- see above for
        mov err_code, eax   //  |  detailed information
        mov eax, cr2        //  |
        mov err_addr, eax   // -+
     end;

     err_code := err_code and 15;     // make sure only bits of interest for us are set

     pde_idx := get_pde_index(err_addr);  // +- used later to update PD
     pte_idx := get_pte_index(err_addr);  // +

     {$ifdef DEBUG}    // output some debugging information
     writeln('Page Fault Exception!');

     writeln('Error code: ' + IntToStr(err_code));
     writeln('Faulting EIP: ' + IntToHexAligned(err_eip, 8));
     writeln('Faulting address: ' + IntToHexAligned(err_addr, 8));

     writeln('Faulting address PDE-index: ' + IntToStr(pde_idx));
     writeln('Faulting address PTE-index: ' + IntToStr(pte_idx));
     {$endif}

     // get new free page
     new_page := get_free_page;

     // out of pages?
     if new_page = nil then
     begin  // we haven't yet implemented swapping so we need to print a nice message
        writeln('no free pages available, swapping not yet implemented, halting cpu');
        asm
           cli
           hlt
        end;
        // btw: shall we use a linux-like swapping partition or
        // shall we implement a windows like swapping file?
     end;

     asm
        mov eax, cr3        // address of PD
        mov ebx, pde_idx    // -+
        shl ebx, 2          //  |
        add eax, ebx        //  +- get address of PT
        mov ebx, [eax]      //  |
        and ebx, $FFFFF000  // -+
        mov eax, pte_idx    // -+
        shl eax, 2          //  |
        add ebx, eax        //  +- set physical address &
        mov eax, new_page   //  |  properties of virtual address
        or eax, 3           //  |  << ! FIX ME ! >> user level!!
        mov [ebx], eax      // -+
        mov eax, cr3        // -+- update PD
        mov cr3, eax        // -+
     end;

     asm
        popad               // restore EAX, ECX, EDX, EBX, ESP, EBP, ESI and EDI
        leave               // -> mov esp, ebp | pop ebp
        add esp, 4          // "delete" error code (-> cpu now knows, we handled the fault
        iret                // return to faulting code
        cli                 // -+
        hlt                 // -+- just to be sure
     end;
end;

{******************************************************************************
 *   get_pde_index
 ******************************************************************************
 *   input:  virtual address
 *   output: page directory index of page table containing this address
 ******************************************************************************}
function get_pde_index( address: longint ): LongInt; [public, alias:'GET_PDE_INDEX'];
var
   tmp : LongInt;
begin
     tmp := address;

     asm
        mov eax, tmp    // -+
        shr eax, 12     //  +- tmp := tmp div 0x1000
        mov tmp, eax    // -+
     end;

     tmp := tmp div $400;

     get_pde_index := tmp;
end;

{******************************************************************************
 *   get_tde_index
 ******************************************************************************
 *   input:  virtual address
 *   output: page table index of page containing given address
 ******************************************************************************}
function get_pte_index( address: longint ): LongInt; [public, alias:'GET_PTE_INDEX'];
var
   tmp, pde, pde_base: LongInt;
   s: string;
begin
     tmp := address;

     pde := get_pde_index(tmp);
     pde_base := (pde * $400);

     asm
        mov eax, pde_base    // -+
        shl eax, 12          //  +- pde_base := pde_base * 0x1000
        mov pde_base, eax    // -+
     end;

     tmp := tmp - pde_base;
     tmp := tmp div $1000;

     get_pte_index := tmp;
end;

{******************************************************************************
 *   get_free_page
 ******************************************************************************
 *   input:  %
 *   output: pointer to a free page (pointer to physical address!)
 *
 *  does:
 *   searches the MEMORY_MAPs for a free page above 16 MB
 ******************************************************************************}
function get_free_page: Pointer; [public, alias:'GET_FREE_PAGE'];
var
   i, mml1_idx: LongInt;
   tmp: Pointer;
begin
     tmp := nil;

     if TOTAL_RAM <= 16384 then  // installed RAM <= 16 MB ??
     begin
        get_free_page := get_free_dma_page; // no difference between dma and "normal" pages
        exit;
     end;

     // first we search in above 16 MB for a free page
     for i := 4 to NO_MME_L2-1 do
     begin
         	// are there any memory map entries, indicating free pages?
         	if scan_bits(MEMORY_MAP_L2[i]) <> 32 then
         	begin
         	        // calculate index for our corresponding memory map level 1
                    mml1_idx := (i * 32) + (31-scan_bits(MEMORY_MAP_L2[i]));

                    // just to be sure
                    if scan_bits(MEMORY_MAP_L1[mml1_idx]) <> 32 then
                    begin // page is free
                            // calc "page-ID" to free page
                            tmp := Pointer(((mml1_idx*32)+(31-scan_bits(MEMORY_MAP_L1[mml1_idx])))*4096);
                            // mark page in bitmap as "not free"
                            MEMORY_MAP_L1[mml1_idx] := set_bit(scan_bits(MEMORY_MAP_L1[mml1_idx]), MEMORY_MAP_L1[mml1_idx]);
                            // all 32 pages of current memory-map-l1 entry in use?
                            if scan_bits(MEMORY_MAP_L1[mml1_idx]) = 32 then
                            // -> set current entry of mml1 as no more pages available
                            MEMORY_MAP_L2[i] := set_bit(scan_bits(MEMORY_MAP_L2[i]), MEMORY_MAP_L2[i]);
                    end;
            end;
            if tmp <> nil then
               break;
      end;

     // if above 16 MB no more pages are free then try to get one below 16MB
     if tmp = nil then
        tmp := get_free_dma_page;

     get_free_page := tmp;
end;

{******************************************************************************
 *   get_free_dma_page
 ******************************************************************************
 *   input:  %
 *   output: Pointer to a free page (pointer to physical address!)
 *
 *  does:
 *   searches MEMORY_MAPs for a free page below 16 MB
 ******************************************************************************}
function get_free_dma_page: Pointer; [public, alias:'GET_FREE_DMA_PAGE'];
var
   i, j, mml1_idx: LongInt;
   tmp: Pointer;
begin
     tmp := nil;

     if TOTAL_RAM > 16384 then  // installed RAM > 16 MB ??
         	j := 3
     else
            j := NO_MME_L2-1;

     for i := 0 to j do
     begin
         	// are there any memory map entries, indicating free pages?
         	if scan_bits(MEMORY_MAP_L2[i]) <> 32 then
         	begin
         	        // calculate index for our corresponding memory map level 1
                    mml1_idx := (i * 32) + (31-scan_bits(MEMORY_MAP_L2[i]));

                    // just to be sure
                    if scan_bits(MEMORY_MAP_L1[mml1_idx]) <> 32 then
                    begin // page is free
                            // calc "page-ID" to free page
                            tmp := Pointer(((mml1_idx*32)+(31-scan_bits(MEMORY_MAP_L1[mml1_idx])))*4096);
                            // mark page in bitmap as "not free"
                            MEMORY_MAP_L1[mml1_idx] := set_bit(scan_bits(MEMORY_MAP_L1[mml1_idx]), MEMORY_MAP_L1[mml1_idx]);
                            // all 32 pages of current memory-map-l1 entry in use?
                            if scan_bits(MEMORY_MAP_L1[mml1_idx]) = 32 then
                            // -> set current entry of mml1 as no more pages available
                            MEMORY_MAP_L2[i] := set_bit(scan_bits(MEMORY_MAP_L2[i]), MEMORY_MAP_L2[i]);
                    end;
            end;
            if tmp <> nil then
               break;
     end;

     get_free_dma_page := tmp;
end;

{******************************************************************************
 *   free_page
 ******************************************************************************
 *   input:  Pointer to Page not used anymore
 *   output: %
 *
 *  does:
 *   calc MEMORY_MAPs index of page and unsets the corresponding bits
 *
 ******************************************************************************}
procedure free_page( page: Pointer ); [public, alias:'FREE_PAGE'];
var
   mml1_idx, mml2_idx, dPage: DWord;
   scan_l1_idx, scan_l2_idx: byte;
begin
     // dpage := page div 4096;
     dpage := DWord(page) shr 12;

     // calculate MEMORY_MAP L1&L2 indizes
     mml1_idx := (dpage div 32);
     scan_l1_idx := 31-((dpage) mod 32);
     mml2_idx := (dpage div 1024);
     scan_l2_idx := 31-((mml1_idx) mod 32);

     // unset bits to indicate pages as free
     MEMORY_MAP_L1[mml1_idx] := unset_bit( scan_l1_idx, MEMORY_MAP_L1[mml1_idx] );
     MEMORY_MAP_L2[mml2_idx] := unset_bit( scan_l2_idx, MEMORY_MAP_L2[mml2_idx] );
end;

{******************************************************************************
 *   nucleOS Virtual Memory Manager
 ******************************************************************************
 *
 * vmm currently consists of three functions:
 *    mem_alloc  allocates memory
 *    sbrk       increases heap
 *    free       deallocates memory allocated with mem_alloc
 *
 * how does it work?
 *
 *    we use two linked lists to have control over the dynamically allocated
 *    memory: alloclist & freelist
 *
 *      alloclist contains allocated memory
 *      freelist contains deallocated memory, free for use
 *
 *    whenever mem_alloc is called, it begins searching freelist  (beginning 
 *    with @freelist). if it finds space equal to requested size it just moves
 *    the link from freelist to alloclist, directly between @alloclist and
 *    alloclist.next. if it finds space greater than requested size+10 it splits
 *    the link and moves the first part of it from freelist to alloclist and changes
 *    the old freelist-link to fit the second part.
 *    if no free space could be found, it just increases brk and uses the new heap
 *    space to handle the request.
 *
 *    when free is called, it searches alloclist for corresponding link and
 *    simply moves the link from alloclist to freelist right after @freelist:
 *    maybe we later look up, if the freed memory is at the end of allocated
 *    heap and decrease heap.
 *
 ******************************************************************************}

function mem_alloc( n: word ): Pointer; [public, alias:'MEM_ALLOC'];
var
   m, g, tmp: Pmem_header;
begin
     // start at @freelist
     m := @freelist;
     repeat
           g := m^.next;
           if (g^.size > n+10) then
           begin  // if there is enough space for the requested amount of bytes
                  // AND a new header AND at least 5 bytes we split the link
               	m^.next := g^.next;                  // remove link from freelist
               	tmp := alloclist.next;               // -+- insert link into
               	alloclist.next := g;                 //  |  alloclist
               	alloclist.next^.next := tmp;         //  |  and
               	alloclist.next^.size := n;           // -+  correct size
               	tmp := freelist.next;                // -+- insert remaining space
               	freelist.next := g+n;                //  |  into freelist
               	freelist.next^.next := tmp;          //  |  and
               	freelist.next^.size := g^.size-n-6;  // -+  correct size
               	mem_alloc := alloclist.next+1;
               	exit;
           end
           else if (g^.size = n) then // if request fits perfectly into old link
           begin
               	m^.next := g^.next;                 // remove link from freelist...
               	tmp := alloclist.next;              // -+- ...and move it to
               	alloclist.next := g;                //  |  alloclist
               	alloclist.next^.next := tmp;        // -+
               	mem_alloc := alloclist.next+1;
               	exit;
           end;
           m := g;
     until m = @freelist; // repeat until we're at the beginning again

     // if we can't find free mem, resize heap and allocate new heap:
     mem_alloc := sbrk(n+6)+6;
     tmp := alloclist.next;               // -+- insert the new link
     alloclist.next := mem_alloc-6;       //  |  into alloclist
     alloclist.next^.next := tmp;         //  |  set following link
     alloclist.next^.size := n;           // -+  set link size
end;

{******************************************************************************
 *   sbrk
 ******************************************************************************
 *   input:  size heap gets incremented in bytes
 *   output: Pointer to old heap end
 *
 *  does:
 *   it simply increases the end of heap by n bytes
 *
 ******************************************************************************}
function sbrk( n: word ): Pointer; [public, alias:'SBRK'];
begin
     sbrk := Pointer(brk);
     brk := brk + n;
end;

{******************************************************************************
 *   free
 ******************************************************************************
 *   input:  Pointer
 *   output: %
 *
 *  does:
 *   free just frees memory allocated using mem_alloc and moves the used link
 *   from alloclist to freelist
 *
 ******************************************************************************}
Procedure free( var p: Pointer ); [public, alias:'FREE'];
var
   m, g, tmp: Pmem_header;
begin
     m := @alloclist;
     repeat
           g := m^.next;
           if (g+1 = p) then
           begin
              m^.next := g^.next;
              tmp := freelist.next;
              freelist.next := g;
              freelist.next^.next := tmp;
              p := nil;
              exit;
           end;
           m := g;
     until m = @alloclist;
end;

{******************************************************************************
 *   mem_copy
 ******************************************************************************
 *   input:  pointer to source and destination, length is the amount in bytes to be copied
 *   output: %
 *
 *  does:
 *   copies length bytes from src to dest
 *
 ******************************************************************************}
procedure mem_copy( src, dest: Pointer; length: LongInt ); assembler; [public, alias:'MEM_COPY'];
asm
   push eax            // -+- save some registers
   push ebx            //  |
   push edx            // -+

   cld

   mov esi, src        // -+- set up esi & edi for movs* instruction
   mov edi, dest       // -+
   mov eax, length     // copy ecx to eax
   xor edx, edx        // clear edx
   mov ebx, 4          // set up divisor
   div ebx             // divide ecx:eax by 4

   mov ecx, eax        // -+- copy 4 bytes at once to save some time 3-)
   rep movsd           // -+

   mov ecx, edx        // -+- copy remaining bytes (max 3)
   rep movsb           // -+

   pop edx             // -+- restore saved registers
   pop ebx             //  |
   pop eax             // -+
end;

{******************************************************************************
 *   getPhysicalAddress
 ******************************************************************************
 *   input:  VirtualAddress specifies virtual address to look up in cr3
 *   output: pointer containing physical address
 *
 *  does:
 *   just looks up page directory given with cr3 for physical address. returns
 *   nil if page is not present
 *
 ******************************************************************************}
function getPhysicalAddress( VirtualAddress, cr3: Pointer ): Pointer; [public, alias:'GETPHYSICALADDRESS'];
begin
     // do nothing at all
end;

begin
end.
